# build_crossover_rho_grids.py
import argparse, os, pathlib, yaml, numpy as np

def load_base_rho(path: str, column: str|None):
    p = pathlib.Path(path)
    if not p.exists():
        raise FileNotFoundError(f"Source not found: {path}")
    if p.suffix.lower() == ".npy":
        arr = np.load(p)
        arr = np.asarray(arr, dtype=float).ravel()
        return arr
    # CSV: try pandas only when needed
    try:
        import pandas as pd
    except ImportError as e:
        raise RuntimeError("Install pandas to read CSV, or provide a .npy file") from e
    df = pd.read_csv(p)
    col = column or ("rho" if "rho" in df.columns else None)
    if col is None:
        raise ValueError(f"CSV needs a 'rho' column or use --column. Columns: {list(df.columns)}")
    arr = df[col].to_numpy(dtype=float).ravel()
    return arr

def resample_rho(rho_base: np.ndarray, n_out: int, prefer_cubic: bool=True) -> np.ndarray:
    x = np.linspace(0.0, 1.0, num=rho_base.size)
    xi = np.linspace(0.0, 1.0, num=n_out)
    if prefer_cubic:
        try:
            from scipy.interpolate import CubicSpline
            cs = CubicSpline(x, rho_base, bc_type=((1,0.0),(1,0.0)))  # clamped, zero slope
            return cs(xi).astype(float)
        except Exception:
            pass  # fall back to linear
    return np.interp(xi, x, rho_base).astype(float)

def main():
    ap = argparse.ArgumentParser(description="Build rho_grid_L{L}.npy for crossover alignment")
    ap.add_argument("--source", required=True, help="Path to base rho sequence (.npy or .csv with a 'rho' column)")
    ap.add_argument("--column", default=None, help="CSV column name for rho (default: 'rho' if present)")
    ap.add_argument("--config", default="configs/default.yaml")
    ap.add_argument("--outdir", default="data/results/vol4_wilson_loop_pipeline_crossover_analysis")
    ap.add_argument("--prefer-cubic", action="store_true", help="Use cubic (clamped) if SciPy available")
    args = ap.parse_args()

    cfg = yaml.safe_load(open(args.config,"r",encoding="utf-8"))
    # L list: prefer adjoint_volume.volumes, fall back to top-level L_values/L_list
    Ls = (cfg.get("adjoint_volume",{}).get("volumes")
          or cfg.get("L_values")
          or cfg.get("L_list"))
    if not Ls:
        raise KeyError("No L list found in YAML (looked for adjoint_volume.volumes, L_values, L_list)")

    rho_base = load_base_rho(args.source, args.column)
    if rho_base.size < 4:
        raise ValueError("Base rho sequence too short (need >= 4 points for spline)")

    outdir = pathlib.Path(args.outdir); outdir.mkdir(parents=True, exist_ok=True)

    for L in sorted(int(L) for L in Ls):
        n_out = 2 * (L**2)
        rho_grid = resample_rho(rho_base, n_out, prefer_cubic=args.prefer_cubic)
        out_path = outdir / f"rho_grid_L{L}.npy"
        np.save(out_path, rho_grid)
        print(f"[OK] Wrote {out_path} (len={rho_grid.size})")

if __name__ == "__main__":
    main()
